#============================================================================== 
# ** Ruby.Math.dist
#------------------------------------------------------------------------------
# Description:
# ------------
# This set allows the generation of discrete distributed random values.
#
# Class List:
# -----------
# Math::Uniform
# Math::Bernoulli
# Math::Binomial
# Math::Geometric
# Math::Poisson
#
# Method List: (avaiable for all classes)
# ------------
# initialize
# generate
# generate_array
# prob
#==============================================================================

MACL::Loaded << 'Ruby.Math.dist'

#============================================================================== 
# ** Math     
#==============================================================================

module Math        
  #-------------------------------------------------------------------------
  # * Distribution precision (keep it high and float)
  #-------------------------------------------------------------------------
  Dist_precision = 1000.0
  #============================================================================== 
  # ** Math::Uniform                                                 By: Tibuda
  #-----------------------------------------------------------------------------
  #  A class for generating unform distributed random values. It's used as a
  #  base for other distributions.                                           
  #-----------------------------------------------------------------------------  
  #  The discrete uniform distribution can be characterized by saying that all
  #  values of a finite set of possible values are equally probable.
  #==============================================================================
  class Uniform        
    #--------------------------------------------------------------------------
    # * Public Instance Variables
    #--------------------------------------------------------------------------
    attr_reader :min
    attr_reader :max
    #-------------------------------------------------------------------------
    # * Name      : Initialize
    #   Info      : Object Initialization
    #   Author    : Tibuda
    #   Call Info : Zero to Two
    #               Integer minumum value (default 0)
    #               Integer maximum value (default 1)
    #-------------------------------------------------------------------------
    def initialize(min = 0, max = 1)
      @min = min
      @max = max
      @max += @min if max < min
    end
    #-------------------------------------------------------------------------
    # * Name      : Generate
    #   Info      : Generates uniform distributed value
    #   Author    : Tibuda
    #   Call Info : No Arguments
    #------------------------------------------------------------------------- 
    def generate
      return @min + rand(@max - @min)
    end 
    def gen
      return generate
    end           
    #-------------------------------------------------------------------------
    # * Name      : Generate array
    #   Info      : Generates an array with uniform distributed values
    #   Author    : Tibuda
    #   Call Info : One
    #               Integer array size
    #------------------------------------------------------------------------- 
    def generate_array(size)
      result = []
      for i in 0...size
        result << generate
      end
      return result
    end  
    def gen_array(size)
      return generate_array(size)
    end
    #-------------------------------------------------------------------------
    # * Name      : Probability mass function
    #   Info      : Returns the probability of observing a value
    #   Author    : Tibuda
    #   Call Info : One
    #               Integer tested value
    #-------------------------------------------------------------------------
    def prob(x)
      return (x < @min || x > @max) ? 0 : 1 / (@max - @min)
    end    
  end     
  
  #============================================================================== 
  # ** Math::Bernoulli                                               By: Tibuda
  #-----------------------------------------------------------------------------  
  #  A class for generating Bernoulli distributed random values. It's used as a
  #  base class for binomial and geometric distributions.        
  #-----------------------------------------------------------------------------  
  #  Bernoulli trial is an experiment whose outcome is random and can be either
  #  of two possible outcomes, "success" and "failure". In practice it refers to
  #  a single experiment which can have one of two possible outcomes, like "yes
  #  or no" questions.
  #==============================================================================
  class Bernoulli < Uniform      
    #--------------------------------------------------------------------------
    # * Public Instance Variables
    #--------------------------------------------------------------------------
    attr_reader :p           
    attr_reader :success        
    attr_reader :failure           
    #-------------------------------------------------------------------------
    # * Name      : Initialize
    #   Info      : Object Initialization
    #   Author    : Tibuda
    #   Call Info : Zero to Three
    #               Float probability of success (default 0.5)
    #               Variant success return value (default 1)
    #               Variant failure return value (default 0)
    #-------------------------------------------------------------------------  
    def initialize(p = 0.5, success = 1, failure = 0)
      super(0, Dist_precision)
      @p = p 
      @success = success
      @failure = failure
    end             
    #-------------------------------------------------------------------------
    # * Name      : Generate
    #   Info      : Generates Bernoulli distributed value
    #   Author    : Tibuda
    #   Call Info : No Arguments
    #------------------------------------------------------------------------- 
    def generate
      return super.to_f / Dist_precision <= @p ? @success : @failure
    end   
    #-------------------------------------------------------------------------
    # * Name      : Probability mass function
    #   Info      : Returns the probability of observing a value
    #   Author    : Tibuda
    #   Call Info : One
    #               Integer tested value
    #-------------------------------------------------------------------------
    def prob(x)
      return @p if x == @success
      return 1 - @p if x == @failure
      return 0
    end     
  end
  
  #============================================================================== 
  # ** Math::Binomial                                                By: Tibuda
  #-----------------------------------------------------------------------------     
  #  A class for generating binomial distributed random values.   
  #-----------------------------------------------------------------------------  
  #  The binomial distribution is the probability distribution of the number of
  #  "successes" in n independent Bernoulli trials, with the same probability of
  #  "success" on each trial.
  #==============================================================================
  class Binomial < Bernoulli               
    #--------------------------------------------------------------------------
    # * Public Instance Variables
    #--------------------------------------------------------------------------
    attr_reader :n                        
    #-------------------------------------------------------------------------
    # * Name      : Initialize
    #   Info      : Object Initialization
    #   Author    : Tibuda
    #   Call Info : Zero to Two
    #               Integer number of trials (default 1)
    #               Float probability of success (default 0.5)
    #-------------------------------------------------------------------------
    def initialize(n = 1, p = 0.5)
      super(p, 1, 0)
      @n = n
    end   
    #-------------------------------------------------------------------------
    # * Name      : Generate
    #   Info      : Generates binomial distributed value
    #   Author    : Tibuda
    #   Call Info : No Arguments
    #------------------------------------------------------------------------- 
    def generate  
      result = 0
      for i in 0...@n
        result += super
      end
      return result
    end
    #-------------------------------------------------------------------------
    # * Name      : Probability mass function
    #   Info      : Returns the probability of observing a value
    #   Author    : Tibuda
    #   Call Info : One
    #               Integer tested value
    #-------------------------------------------------------------------------
    def prob(x)
      binomial_coef = @n.factorial / (x.factorial * (@n - x).factorial)
      return  binomial_coef * @p ** x * (1 - @p) ** (@n - x)
    end
  end
  
  #============================================================================== 
  # ** Math::Geometric                                               By: Tibuda
  #-----------------------------------------------------------------------------     
  #  A class for generating geometric distributed random values.   
  #-----------------------------------------------------------------------------  
  #  The geometric distribution is the probability distribution of the number
  #  of Bernoulli trials needed to get one success.
  #==============================================================================
  class Geometric < Bernoulli   
    #-------------------------------------------------------------------------
    # * Name      : Initialize
    #   Info      : Object Initialization
    #   Author    : Tibuda
    #   Call Info : Zero to One
    #               Float probability of success (default 0.5)
    #-------------------------------------------------------------------------
    def initialize(p = 0.5)
      super(p, true, false)
    end         
    #-------------------------------------------------------------------------
    # * Name      : Generate
    #   Info      : Generates geometric distributed value
    #   Author    : Tibuda
    #   Call Info : No Arguments
    #------------------------------------------------------------------------- 
    def generate
      k = 0
      while super == false
        k += 1
      end
      return k
    end         
    #-------------------------------------------------------------------------
    # * Name      : Probability mass function
    #   Info      : Returns the probability of observing a value
    #   Author    : Tibuda
    #   Call Info : One
    #               Integer tested value
    #-------------------------------------------------------------------------
    def prob(x)
      return @p * (1 - @p) ** x
    end
  end
  
  #============================================================================== 
  # ** Math::Poisson                            By: Tibuda using Knuth algorithm
  #-----------------------------------------------------------------------------     
  #  A class for generating Poisson distributed random values.   
  #-----------------------------------------------------------------------------  
  #  The Poisson distribution expresses the probability of a number of events
  #  occurring in a fixed period of time if these events occur with a known
  #  average rate, and are independent of the time since the last event.
  #==============================================================================
  class Poisson < Uniform             
    #--------------------------------------------------------------------------
    # * Public Instance Variables
    #--------------------------------------------------------------------------
    attr_reader :lambda     
    #-------------------------------------------------------------------------
    # * Name      : Initialize
    #   Info      : Object Initialization
    #   Author    : Tibuda
    #   Call Info : One
    #               Float expected number of occurrences
    #-------------------------------------------------------------------------
    def initialize(lambda)    
      super(0, Dist_precision)
      @lambda = lambda
    end                
    #-------------------------------------------------------------------------
    # * Name      : Generate
    #   Info      : Generates Poisson distributed value
    #   Author    : Tibuda using Knuth algorithm
    #   Call Info : No Arguments
    #------------------------------------------------------------------------- 
    def generate
      k = 0
      p = 1.0
      while p >= Math.exp(-@lambda)
        k += 1
        p *= super.to_f / Dist_precision
      end
      return k - 1
    end       
    #-------------------------------------------------------------------------
    # * Name      : Probability mass function
    #   Info      : Returns the probability of observing a value
    #   Author    : Tibuda
    #   Call Info : One
    #               Integer tested value
    #-------------------------------------------------------------------------
    def prob(x)
      return Math.exp(-@lambda) * @lambda ** x / x.factorial
    end
  end
end